﻿using Hims.Api.Models;
using Hims.Api.Utilities;
using Hims.Domain.Helpers;
using Hims.Domain.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Hims.Shared.UserModels.Common;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{
    /// <inheritdoc />
    [Route("api/diet")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class DietSlotMasterController : BaseController
    {
        /// <summary>
        /// The auditlog services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <summary>
        /// The auditlog services.
        /// </summary>
        private readonly IAESHelper aeshelper;


        private readonly IDietSlotsMasterService dietSlotsMasterService;

        public DietSlotMasterController(
           IAuditLogService auditLogServices,
           IDietSlotsMasterService dietSlotsMasterServices,
           IAESHelper aeshelper)
        {

            this.auditLogServices = auditLogServices;
            this.dietSlotsMasterService = dietSlotsMasterServices;
            this.aeshelper = aeshelper;
        }

        /// <summary>
        /// The insert slot async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("add-shift")]
        public async Task<ActionResult> InsertAsync([FromBody] DietSlotsModel model, [FromHeader] LocationHeader header)
        {
            model = (DietSlotsModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var response = await this.dietSlotsMasterService.InsertSlotsAsync(model).ConfigureAwait(false);

            if (response > 0)
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.DietSlots,
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $@"{model.CreatedByName} has added a new slot '{model.StartTime}-{model.EndTime}' on {DateTime.UtcNow.AddMinutes(330)}",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }

            return this.Success(response);

        }



        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchShiftSlotsAsync([FromBody] DietSlotsModel model)
        {           
            var response = await this.dietSlotsMasterService.FetchSlotsAsync(model);
            return this.Success(response);

        }

        [HttpPost]
        [Route("update-shift")]
        public async Task<ActionResult> UpdateAsync([FromBody] DietSlotsModel model, [FromHeader] LocationHeader header)
        {
            model = (DietSlotsModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(header.LocationId);
            var response = await this.dietSlotsMasterService.UpdateAsync(model);
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.DietSlots,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LogDescription = $"{model.CreatedByName} has updated <b>Shift Name</b> of <strong>{model.Name}</strong> successfully.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLog);
            }

            return this.Success(response);
        }


        [HttpPost]
        [Route("modify-status")]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] DietSlotsModel model, [FromHeader] LocationHeader header)
        {

            model = (DietSlotsModel)EmptyFilter.Handler(model);
            var response = await this.dietSlotsMasterService.ModifyStatusAsync(model);
            if (response > 0)
            {                
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.DietSlots,
                        LogDate = DateTime.Now,
                        LogFrom = short.Parse(model.LoginRoleId.ToString()),
                        LogDescription = model.Active == true ? $@"{model.ModifiedByName} has Activated status of shift <br/>name: '{model.Name}' on {DateTime.UtcNow.AddMinutes(330)}" : $@"{model.ModifiedByName} has Deactivated status of shift <br/>name: '{model.Name}' on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);               
            }
            return this.Success(response);

        }


    }
}
